#!/usr/bin/env python3
"""
Snake Game for Raspberry Pi 500+ RGB Keyboard
Real-time version with pynput for instant key detection
Install: pip install pynput
"""

from RPiKeyboardConfig import RPiKeyboardConfig
import random
import time
from pynput import keyboard

# Initialize keyboard
kbd = RPiKeyboardConfig()

# Get ALL valid LED positions - use the entire keyboard!
valid_positions = set()
all_leds = kbd.get_leds()
for led in all_leds:
    valid_positions.add(tuple(led.matrix))

# Define playable area - Rows 1-5 only (exclude F-keys row 0)
PLAY_AREA = []
for row in range(1, 6):  # Rows 1-5 only
    for col in range(16):  # All columns
        if (row, col) in valid_positions:
            PLAY_AREA.append((row, col))

print(f"Playable area: {len(PLAY_AREA)} positions (entire keyboard)")

# Colors (HSV format)
COLORS = {
    'snake': (85, 255, 255),    # Green
    'food': (170, 255, 255),     # Cyan
    'score': (42, 255, 200),     # Orange
}

# Directions
DIRECTIONS = {
    'w': (-1, 0),  # Up
    's': (1, 0),   # Down
    'a': (0, -1),  # Left
    'd': (0, 1),   # Right
}

class SnakeGame:
    def __init__(self):
        start_col = 7
        start_row = 3
        self.snake = [(start_row, start_col), (start_row, start_col-1), (start_row, start_col-2)]
        self.direction = (0, 1)  # Moving right
        self.next_direction = (0, 1)
        self.food = None
        self.score = 0
        self.game_over = False
        self.running = True
        
    def spawn_food(self):
        """Spawn food in a random empty location"""
        # Only spawn in valid positions that are also in PLAY_AREA
        empty_spaces = [pos for pos in PLAY_AREA if pos not in self.snake and pos in valid_positions]
        if empty_spaces:
            self.food = random.choice(empty_spaces)
        else:
            print("\n⚠️  Warning: No empty spaces for food!")
    
    def move(self):
        """Move the snake in current direction"""
        if self.game_over:
            return False
        
        # Update direction (allows queuing one move ahead)
        self.direction = self.next_direction
        
        # Calculate new head position
        head = self.snake[0]
        new_head = (head[0] + self.direction[0], head[1] + self.direction[1])
        
        # If new position doesn't exist, keep moving in that direction until we find a valid one
        attempts = 0
        max_attempts = 20  # Prevent infinite loops
        while new_head not in PLAY_AREA and attempts < max_attempts:
            new_head = (new_head[0] + self.direction[0], new_head[1] + self.direction[1])
            attempts += 1
        
        # If we still can't find a valid position after max attempts, hit a wall
        if new_head not in PLAY_AREA:
            self.game_over = True
            return False
        
        # Check self collision
        if new_head in self.snake:
            self.game_over = True
            return False
        
        # Add new head
        self.snake.insert(0, new_head)
        
        # Check if food eaten
        if new_head == self.food:
            self.score += 1
            self.spawn_food()
        else:
            self.snake.pop()
        
        return True
    
    def queue_direction(self, new_direction):
        """Queue next direction change (prevent 180 degree turns)"""
        if new_direction:
            opposite = (-self.direction[0], -self.direction[1])
            if new_direction != opposite:
                self.next_direction = new_direction
                return True
        return False
    
    def draw(self):
        """Draw the game state to the keyboard"""
        kbd.rgb_clear()
        
        # Draw snake body
        for pos in self.snake[1:]:
            if pos in valid_positions:
                kbd.set_led_by_matrix(matrix=list(pos), colour=(85, 255, 150))
        
        # Draw snake head
        if self.snake and self.snake[0] in valid_positions:
            kbd.set_led_by_matrix(matrix=list(self.snake[0]), colour=COLORS['snake'])
        
        # Draw food
        if self.food and self.food in valid_positions:
            kbd.set_led_by_matrix(matrix=list(self.food), colour=COLORS['food'])
        
        kbd.send_leds()

# Global game instance
game = None

def on_press(key):
    """Handle key press events"""
    global game
    if game is None:
        return
    
    try:
        # Get the character
        if hasattr(key, 'char') and key.char:
            char = key.char.lower()
            
            if char == 'q':
                print("\n\nQuitting...")
                game.running = False
                return False  # Stop listener
            
            elif char in DIRECTIONS:
                if game.queue_direction(DIRECTIONS[char]):
                    print(f"\r🎮 Direction: {char.upper():<20}", end='', flush=True)
    
    except AttributeError:
        pass

def main():
    global game
    
    print("\n" + "=" * 60)
    print("🐍 SNAKE GAME - Real-time Mode")
    print("=" * 60)
    print("The snake moves automatically!")
    print("Controls (instant response, no Enter needed!):")
    print("  W = up | A = left | S = down | D = right | Q = quit")
    print("=" * 60)
    print(f"Play area: {len(PLAY_AREA)} keys (entire keyboard below F-keys)")
    
    # Debug: show which positions are valid
    print("\nChecking for problematic positions...")
    row_1_positions = [pos for pos in PLAY_AREA if pos[0] == 1]
    print(f"Row 1 positions: {sorted(row_1_positions, key=lambda x: x[1])}")
    
    input("\nPress Enter to start...")
    
    play_again = True
    
    while play_again:
        # Initialize game
        game = SnakeGame()
        game.spawn_food()
        kbd.set_led_direct_effect()
        game.draw()
        
        print("\n🐍 Game started! Snake is moving...")
        print("Press W/A/S/D to change direction (Q to quit)")
        print("-" * 60)
        
        # Start keyboard listener
        listener = keyboard.Listener(on_press=on_press)
        listener.start()
        
        # Game loop
        last_move_time = time.time()
        move_delay = 0.35  # Seconds between moves
        
        try:
            while game.running and not game.game_over:
                current_time = time.time()
                
                # Move snake at fixed intervals
                if current_time - last_move_time >= move_delay:
                    if game.move():
                        game.draw()
                        print(f"\r📊 Score: {game.score} | Length: {len(game.snake)} | Position: {game.snake[0]}     ", end='', flush=True)
                        last_move_time = current_time
                
                time.sleep(0.01)
        
        except KeyboardInterrupt:
            print("\n\n⚠️  Game interrupted!")
            listener.stop()
            break
        
        # Stop listener
        listener.stop()
        
        # Game over - auto restart
        if game.game_over:
            print(f"\n\n💀 GAME OVER! Score: {game.score} | Length: {len(game.snake)}")
            
            # Flash red
            for _ in range(2):
                kbd.rgb_clear()
                for pos in PLAY_AREA[::3]:
                    if pos in valid_positions:
                        kbd.set_led_by_matrix(matrix=list(pos), colour=(0, 255, 255))
                kbd.send_leds()
                time.sleep(0.25)
                kbd.rgb_clear()
                kbd.send_leds()
                time.sleep(0.25)
            
            print("Restarting in 2 seconds...")
            time.sleep(2)
            # Loop continues, game restarts
        else:
            play_again = False
    
    # Clean up
    kbd.rgb_clear()
    kbd.send_leds()
    print("\n✨ Thanks for playing! 🎮\n")

if __name__ == "__main__":
    main()